import logging

import toml
from pydantic import BaseModel, ValidationError

from kyd_dataspec_gen.logging_config import setup_logging

setup_logging()
logger = logging.getLogger(__name__)


class Config(BaseModel):
    """
    Configuration class for the kyd_dataspec_gen package.
    This class is used to define the configuration settings for the package.
    """

    # Separators used in values
    separator: str = ";"
    # The keyword used for identifying address columns
    address_column_keyword: str = "address"
    # The keyword used for identifying id column
    identification_column_keyword: str = "id"
    # The upper limit of the number of categories to be considered as categorical
    categorical_limit: int = 250
    # The upper limit of the number of data profile shapes showing in the report
    data_profile_shapes_limit: int = 20

    # For reference dataset matching
    # Published reference dataset id prefix
    published_ref_id_prefix: str = "PREF-"
    # Threshold for fuzzy matching reference data names
    ref_data_fuzzy_threshold: int = 90
    # Limit the number of sample values to include in the published reference dataset
    sample_values_limit: int = 5


def get_config(config_file: str) -> Config:
    """
    Function to get the configuration settings for the kyd_dataspec_gen package.
    This function returns an instance of the Config class.
    """
    config = None
    try:
        with open(config_file, "r") as file:
            try:
                toml_content = toml.load(file)["dataspec"]
                config = Config(
                    separator=toml_content["separator"],
                    address_column_keyword=toml_content["address_column_keyword"],
                    identification_column_keyword=toml_content[
                        "identification_column_keyword"
                    ],
                    categorical_limit=toml_content["categorical_limit"],
                    data_profile_shapes_limit=toml_content["data_profile_shapes_limit"],
                    published_ref_id_prefix=toml_content["published_ref_id_prefix"],
                    ref_data_fuzzy_threshold=toml_content["ref_data_fuzzy_threshold"],
                    sample_values_limit=toml_content["sample_values_limit"],
                )
            except ValidationError as e:
                logger.error("Invalid configuration file: %s", e.json())
    except FileNotFoundError:
        logger.error(
            f"Configuration file '{config_file}' not found. Using default settings."
        )

    if config is None:
        logger.error("Failed to load configuration. Using default settings.")
        config = Config()

    return config
