import logging
from pathlib import Path

import jinja2
import re


logger = logging.getLogger(__name__)


def camel_case_to_title_case(name: str) -> str:
    """
    Converts a camelCase string to Title Case.

    Args:
        name (str): The camelCase string to be converted.

    Returns:
        str: The converted Title Case string.
    """
    individual_str = re.findall(r"[a-z]+|[A-Z][a-z]*", name)
    return " ".join(individual_str).title()


def render_jinja(data: dict, template: str, output_path: str) -> None:
    """
    Renders a Jinja2 template with the provided data and writes the result to an output file.

    Args:
        data (dict): A dictionary containing the data to be used for rendering the template.
        template (str): The name of the Jinja2 template file to be rendered.
        output_path (str): The file path where the rendered output will be written.

    Returns:
        None
    """
    parent_dir = Path(__file__).parent
    template_dir = parent_dir / "templates"
    if "/" in template:
        template_dir = Path(template).parent
        template = Path(template).name
    env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_dir))
    env.filters["camel_case_to_title_case"] = camel_case_to_title_case
    result = env.get_template(template).render(dataspec=data)
    Path(output_path).parent.mkdir(parents=True, exist_ok=True)
    logger.debug(f"Writing {output_path}")
    with open(output_path, "wt", encoding="utf-8") as f:
        f.write(result)
