import logging
import os
from typing import Any, Optional

from dotenv import load_dotenv
from google import genai
from google.genai.errors import ClientError
from google.genai.types import GenerateContentConfigDict


load_dotenv()
logger = logging.getLogger(__name__)
GOOGLE_AI_KEY = os.getenv("GOOGLE_AI_KEY")
GEMINI_MODEL = os.getenv("GEMINI_MODEL", "gemini-2.5-flash-preview-04-17")


gemini_config: GenerateContentConfigDict = {
    # the randomness of the output, higher values mean more random
    "temperature": 0,
    # changes how the model selects tokens for output. Tokens are selected from the most to least probable until the sum of their probabilities equals the topP value.
    "top_p": 0.95,
    # The number of tokens to consider when generating the next token. A higher value means more tokens are considered, leading to more diverse outputs.
    "top_k": 20,
    # The number of responses to generate. A value of 1 means only one response will be generated.
    "candidate_count": 1,
    # introduces a random seed to initialize the LLM's sampling process, ensuring varied outputs for each run. A null value generates a new seed for each run.
    "seed": 3,
    # The maximum number of tokens to generate in the response. A value of 100 means the model will generate up to 100 tokens.
    "max_output_tokens": 100000,
    # A specific sequence of characters or words that signals the model to stop generating text.The strings are case-sensitive.
    "stop_sequences": ["STOP!", "stop"],
    # Positive values penalize tokens that already appear in the generated text, increasing the probability of generating more diverse content.
    "presence_penalty": 0.0,
    # Positive values penalize tokens that repeatedly appear in the generated text, decreasing the probability of repeating content.
    "frequency_penalty": 0.0,
}


def setup_gemini_client():
    """Set up the AI client using the Google API key."""
    try:
        ai_client = genai.Client(api_key=GOOGLE_AI_KEY)
        logger.debug("AI client setup successfully.")
        return ai_client
    except ClientError as e:
        logger.error(f"Error setting up AI client: {e}")
        return None


def generate_response(
    ai_client: genai.Client,
    content: str,
    config: Optional[GenerateContentConfigDict] = None,
) -> Any:
    # Create a local copy of gemini_config to avoid modifying the global config
    local_config = gemini_config.copy()  # Create a local copy of gemini_config

    if config:
        local_config.update(config)
    try:
        response = ai_client.models.generate_content(
            model=GEMINI_MODEL,
            contents=content,
            config=local_config,
        )
        return response.parsed
    except Exception as e:
        logger.warning(
            f"Error generating response: {e}, no description generated, returning a empty dict"
        )
        return {}
