import argparse
import logging
import subprocess
from datetime import datetime
from pathlib import Path

from kyd_dataspec_gen.logging_config import setup_logging

setup_logging()
logger = logging.getLogger(__name__)


def main():
    """Loads Data Specification files into Solidatus."""
    logger.info("Starting data specification generation and loading process...")
    date = datetime.today().strftime("%Y%m%d")
    parser = argparse.ArgumentParser(
        prog="load_kyd_dataspec",
        description="Loads Data Specification files into Solidatus",
    )
    parser.add_argument(
        "-mn",
        "--model-name",
        type=str,
        default=f"dataspec_{date}",
        help="Solidatus model name",
    )
    parser.add_argument(
        "-mt",
        "--model-type",
        choices=["LineageModel", "ReferenceModel"],
        required=True,
        help="Solidatus model type",
    )
    parser.add_argument(
        "-specs",
        "--specification-files-folder",
        default="data_spec/",
        help="Folder of Data Specification files to be loaded to Solidatus",
    )
    parser.add_argument(
        "-c",
        "--create",
        help="Add if you want a file created",
        action="store_true",
    )
    args = parser.parse_args()
    specs_files = [
        str(file) for file in Path(args.specification_files_folder).glob("*.json")
    ]
    logger.debug(f"Loading {len(specs_files)} data specification files into Solidatus")

    command = [
        "kyd-sol-dataspec",
        "-mn",
        args.model_name,
        "-mt",
        args.model_type,
        "-ln",
        "Data-Specifications",
        "-specs",
        *specs_files,
    ]

    if args.create:
        command.append("-c")

    try:
        result = subprocess.run(
            command,
            check=True,
            capture_output=True,
            text=True,
        )
        logger.debug(result.stdout)
    except subprocess.CalledProcessError as e:
        logger.error(f"Command failed with exit code {e.returncode}")
        logger.error(f"Error output: {e.stderr}")
        raise


if __name__ == "__main__":
    main()
