"""Configuration settings for the DOCX to Markdown conversion."""

from typing import TYPE_CHECKING

from .docx_parts_numbering import ListNumberTracker, NumberingParts

if TYPE_CHECKING:
    from lxml import etree

    from .docx_parts_styles import StyleParts


class DocxRuntime:
    """Tracks the current state during DOCX processing."""

    def __init__(self) -> None:
        """Initialize the runtime state."""
        self.current_file: str | None = None
        self.current_paragraph: int = 0
        self.current_run: int = 0

        self.hyperlinks = {}
        self.images = {}
        self.missing_types = {}
        self.footnotes = []

        self.file_list: list[str] = []
        self.list_tracker: ListNumberTracker | None = None
        self.numbering_parts: NumberingParts = NumberingParts(None)
        self.styles_parts: StyleParts | None = None
        self.document_xml: etree._Element | None = None
        self.footnotes_xml: etree._Element | None = None
        self.embedded_tbl_count: int = 0

    def __repr__(self) -> str:
        """Return a string representation of the DocxRuntime object."""
        return (
            f"DocxRuntime(current_file={self.current_file}, "
            f"current_paragraph={self.current_paragraph}, "
            f"current_run={self.current_run}, "
            f"hyperlinks={list(self.hyperlinks.keys())}, "
            f"images={list(self.images.keys())}, "
            f"missing_types={list(self.missing_types.keys())}, "
            f"footnotes={self.footnotes}, "
            f"file_list={self.file_list}, "
            f"numbering_parts={self.numbering_parts}, "
            f"styles_parts={self.styles_parts}, "
            f"document_xml={self.document_xml is not None}, "
            f"footnotes_xml={self.footnotes_xml is not None})"
        )


class DocxConfig:
    """Configuration settings for the DOCX to Markdown conversion."""

    def __init__(self) -> None:
        """Initialize the configuration with default settings."""
        self.ascii_only: bool = False
        self.output_image_dir: str = "images"
        self.suppress_colors: list[str] = ["FFFFFF", "000000", "auto"]
        self.export_images: bool = True
        self.styles: dict = {}
        self.remove_wrapping_tables: list[tuple[int, int]] = []
        self.runtime: DocxRuntime = DocxRuntime()

    def __repr__(self) -> str:
        """Return a string representation of the DocxConfig object."""
        return (
            f"DocxConfig(ascii_only={self.ascii_only}, "
            f"output_image_dir='{self.output_image_dir}', "
            f"suppress_colors={self.suppress_colors}, "
            f"export_images={self.export_images}, "
            f"remove_wrapping_tables={self.remove_wrapping_tables}, "
            f"styles={self.styles}, "
            f"runtime={self.runtime})"
        )
