"""Open and decode raw DOCX files for processing and query"""

import zipfile
from pathlib import Path

from lxml import etree

from .docx_config import DocxConfig
from .docx_constants import (
    DOCUMENT_XML_PATH,
    FOOTNOTES_XML_PATH,
    NUMBERING_XML_PATH,
    STYLES_XML_PATH,
)
from .docx_parts_numbering import NumberingParts
from .docx_parts_styles import StyleParts


class DocxFile:
    """Raw DOCX file representation."""

    def __init__(
        self,
        config: DocxConfig,
        input_path: str | Path | None = None,
    ) -> None:
        """
        Initialize the DocxRaw object with the path to the DOCX file.

        Args:
            input_path (str): The path to the DOCX file.
            config (DocxConfig): The configuration settings.

        """
        self.config = config
        runtime_config = config.runtime

        if input_path is not None:
            runtime_config.current_file = str(input_path)
        runtime_config.file_list = []

        if runtime_config.current_file is None:
            raise ValueError("No input file specified")

        # Create a new zip file to write to
        with zipfile.ZipFile(runtime_config.current_file, "r") as original_zip:
            # Get the list of files in the original document
            runtime_config.file_list = original_zip.namelist()

            if DOCUMENT_XML_PATH in runtime_config.file_list:
                xml_data = original_zip.read(DOCUMENT_XML_PATH)
                self.document_xml = etree.fromstring(xml_data)

            if NUMBERING_XML_PATH in runtime_config.file_list:
                xml_data = original_zip.read(NUMBERING_XML_PATH)
                runtime_config.numbering_parts = NumberingParts(
                    etree.fromstring(xml_data),
                )

            if STYLES_XML_PATH in runtime_config.file_list:
                xml_data = original_zip.read(STYLES_XML_PATH)
                runtime_config.styles_parts = StyleParts(
                    etree.fromstring(xml_data),
                )

            if FOOTNOTES_XML_PATH in runtime_config.file_list:
                xml_data = original_zip.read(FOOTNOTES_XML_PATH)
                runtime_config.footnotes_xml = etree.fromstring(xml_data)

    def __repr__(self) -> str:
        """Return a string representation of the DocxRaw instance."""
        return f"DocxRaw(file_name={self.config.runtime.current_file}, numbering={self.config.runtime.numbering_parts}, styles={self.config.runtime.styles_parts}, footnotes={self.config.runtime.footnotes_xml})"
