"""Handle the numbering.xml file within a Word document."""

# Note if pylance is erroring on this pip install lxml-stubs
from lxml import etree

from .docx_constants import (
    docx_namespaces,
)
from .docx_utils import get_xml_child, get_xml_child_int
from .numbered_item_tracker import ListNumberTracker, TrackerType


class AbsStyleTracker:
    """Track list numbering and the style level."""

    def __init__(self, tracker: ListNumberTracker, level: int) -> None:
        """
        Initialize the AbsStyleTracker.

        Args:
            tracker (ListNumberTracker): The list number tracker instance.
            level (int): The style level.

        """
        self.tracker = tracker
        self.style_level = level


class NumberingParts:
    """Process the numbering parts within the numbering.xml file."""

    def __init__(self, numbering_xml: etree._Element | None) -> None:
        """
        Initialize with the numbering XML element.

        Args:
            numbering_xml (etree.Element): The XML element representing the numbering information.

        """
        self.numbering_xml = numbering_xml

        """Dictionary mapping abstract numbering IDs to their definitions."""
        self.abstract_num_dict: dict[int, DocxAbstractNumbering] = {}
        self.num_dict: dict[int, DocxAbstractNumbering] = {}
        self.style_num_dict: dict[str, AbsStyleTracker] = {}
        self.header_tracker: dict[int, ListNumberTracker] = {}

        if numbering_xml is None:
            return

        if numbering_xml is not None:
            # Find all abstract numbers and create dictionary
            for abstract_num in numbering_xml.findall(
                ".//w:abstractNum",
                namespaces=docx_namespaces,
            ):
                abstract_num_id = abstract_num.attrib.get(
                    f"{{{docx_namespaces['w']}}}abstractNumId",
                )
                if abstract_num_id is not None:
                    abstract_num_id = int(abstract_num_id)
                    self.abstract_num_dict[abstract_num_id] = DocxAbstractNumbering(
                        abstract_num,
                    )

            # Find all absolute reference instances
            for abstract_num in numbering_xml.findall(
                ".//w:num",
                namespaces=docx_namespaces,
            ):
                num_id = abstract_num.attrib.get(
                    f"{{{docx_namespaces['w']}}}numId",
                )
                if num_id is not None:
                    num_id = int(num_id)
                    abstract_num_id = get_xml_child_int(abstract_num, "w:abstractNumId")
                    if abstract_num_id is not None:
                        self.num_dict[num_id] = self.abstract_num_dict[abstract_num_id]

            # Extract all styles and their corresponding abstract numbering levels
            for num_instance, number_fmt in self.num_dict.items():
                self.header_tracker[num_instance] = ListNumberTracker(
                    TrackerType.TEMPLATE_LEVEL,
                    number_fmt.level_types,
                    number_fmt.template_formats,
                    number_fmt.level_start_numbers,
                )

                # Search levels for styles - if found point to header_tracker
                for level in number_fmt.levels.values():
                    if level.style and level.style not in self.style_num_dict:
                        self.style_num_dict[level.style] = AbsStyleTracker(
                            self.header_tracker[num_instance],
                            level.num_level,
                        )

    def get_numbered_text_by_id(
        self,
        num_id: int,
        level: int,
        text: str | None,
    ) -> str:
        """
        Get the number for a specific numbered style by its ID, and render text if supplied.

        Args:
            num_id (int): The ID of the numbered style.
            level (int): The level of the numbered style.
            text (str | None): The text of the numbered style, if available.

        """
        if num_id in self.header_tracker:
            list_tracker = self.header_tracker[num_id]

            if list_tracker:
                text = list_tracker.get_number(
                    level,
                    text,
                )

        return text or ""

    def get_numbered_text_by_style(
        self,
        style_name: str,
        text: str | None,
    ) -> str:
        """
        Get the number for a specific numbered style, and render text if supplied

        Args:
            style_name (str): The style of the numbered style.
            text (str | None): The text of the numbered style, if available.

        """
        if style_name in self.style_num_dict:
            list_tracker = self.style_num_dict[style_name]

            if list_tracker:
                text = list_tracker.tracker.get_number(
                    list_tracker.style_level,
                    text,
                )

        return text or ""

    def __repr__(self) -> str:
        """Return a string representation of the NumberingParts instance."""
        return f"NumberingParts(abs_num_count={len(self.abstract_num_dict)}/num_count={len(self.num_dict)})"


class DocxAbstractNumberingLevel:
    """Encapsulate the abstract numbering information within the numbering.xml file."""

    def __init__(self, numbering_level_xml: etree._Element) -> None:
        """
        Initialize with the abstract numbering information.

        Args:
            numbering_level_xml (etree.Element): The XML element representing the numbering information.

        """
        self.level_xml = numbering_level_xml

        # ilvl is an attribute on the w:lvl element
        self.num_level = int(
            numbering_level_xml.attrib.get(
                f"{{{docx_namespaces['w']}}}ilvl",
                "0",
            ),
        )

        # Find child <w:start w:val="..."/> and read the w:val attribute
        self.start_num = get_xml_child_int(numbering_level_xml, "w:start")
        self.num_fmt = get_xml_child(numbering_level_xml, "w:numFmt")
        self.level_text = get_xml_child(numbering_level_xml, "w:lvlText")
        self.style = get_xml_child(numbering_level_xml, "w:pStyle")

    def __repr__(self) -> str:
        """Return a string representation of the DocxAbstractNumberingLevel instance."""
        return f"DocxAbstractNumberingLevel(num_level={self.num_level}, start_num={self.start_num}, num_fmt={self.num_fmt}, level_text={self.level_text}, style={self.style})"


class DocxAbstractNumbering:
    """Encapsulate the abstract numbering information within the numbering.xml file."""

    def __init__(self, numbering_xml: etree._Element) -> None:
        """
        Initialize with the abstract numbering information.

        Args:
            numbering_xml (etree.Element): The XML element representing the numbering information.

        """
        self.numbering_xml = numbering_xml
        self.abstract_num_id = int(
            numbering_xml.attrib.get(
                f"{{{docx_namespaces['w']}}}abstractNumId",
                "0",
            ),
        )

        # Find all levels within the abstract numbering
        self.levels: dict[int, DocxAbstractNumberingLevel] = {}
        for level_xml in numbering_xml.findall(
            ".//w:lvl",
            namespaces=docx_namespaces,
        ):
            level = DocxAbstractNumberingLevel(level_xml)
            self.levels[level.num_level] = level

        # Generate list of template formats
        self.template_formats: list[str] = []
        self.level_start_numbers: list[int] = []
        self.level_types: list[str] = []
        for level in self.levels.values():
            self.template_formats.append(level.level_text or "")
            self.level_types.append(level.num_fmt or "")
            self.level_start_numbers.append(level.start_num or 1)

    def __repr__(self) -> str:
        """Return a string representation of the DocxAbstractNumbering instance."""
        return f"DocxAbstractNumbering(abstract_num_id={self.abstract_num_id}, levels={len(self.levels)})"
