"""Represents a part of a DOCX document"""

# Note if pylance is erroring on this pip install lxml-stubs
import logging

from lxml import etree

from .docx_constants import (
    docx_namespaces,
)
from .docx_utils import get_xml_child, get_xml_child_int

logger = logging.getLogger(__name__)


class RunPart:
    """Represents a part of a run part in a DOCX document."""

    def __init__(self, run: etree._Element) -> None:
        """Initialize the RunPart."""
        self.part = run
        self.text = run.text or ""
        self.bold = False
        self.italic = False
        self.underline = False
        self.strike = False
        self.color = None
        self.superscript = False
        self.subscript = False

        text_node = run.find(".//w:t", namespaces=docx_namespaces)
        if text_node is not None:
            self.text = text_node.text or ""
        rPr = run.find(".//w:rPr", namespaces=docx_namespaces)
        if rPr is not None:
            self.bold = rPr.find(".//w:b", namespaces=docx_namespaces) is not None
            self.italic = rPr.find(".//w:i", namespaces=docx_namespaces) is not None
            self.underline = rPr.find(".//w:u", namespaces=docx_namespaces) is not None
            self.strike = (
                rPr.find(".//w:strike", namespaces=docx_namespaces) is not None
            )
            self.color = get_xml_child(rPr, "w:color") or ""

            # vert_align = rPr.find(".//w:vertAlign", namespaces=docx_namespaces)
            vert_align = get_xml_child(rPr, ".//w:vertAlign")

            if vert_align is not None:
                self.superscript = vert_align == "superscript"
                self.subscript = vert_align == "subscript"
            else:
                # Some documents use w:position for raised/lowered text.
                pos_element = get_xml_child_int(rPr, ".//w:position")

                if pos_element is not None:
                    # positive -> raised, negative -> lowered
                    if pos_element > 0:
                        self.superscript = True
                    elif pos_element < 0:
                        self.subscript = True

        # Look for footnotes
        self.footnote_reference = get_xml_child(run, ".//w:footnoteReference", "id")
        self.endnote_reference = get_xml_child(run, ".//w:endnoteReference", "id")

    def __repr__(self) -> str:
        """Represent the RunPart."""
        styling = []
        if self.bold:
            styling.append("bold")
        if self.italic:
            styling.append("italic")
        if self.underline:
            styling.append("underline")
        if self.strike:
            styling.append("strike")
        if self.superscript:
            styling.append("superscript")
        if self.subscript:
            styling.append("subscript")
        if self.color:
            styling.append(f"color={self.color}")
        if self.footnote_reference:
            styling.append(f"footnote_reference={self.footnote_reference}")
        if self.endnote_reference:
            styling.append(f"endnote_reference={self.endnote_reference}")
        return f"RunPart(fmt=[{', '.join(styling)}], text={self.text})"
