"""Handle the numbering.xml file within a Word document."""

# Note if pylance is erroring on this pip install lxml-stubs
from lxml import etree

from .docx_constants import (
    docx_namespaces,
)
from .docx_utils import get_xml_child_int


class StyleParts:
    """Process the style parts within the styles.xml file."""

    def __init__(self, styles_xml: etree._Element) -> None:
        """
        Initialize with the styles XML element.

        Args:
            styles_xml (etree.Element): The XML element representing the styles information.

        """
        self.styles_xml = styles_xml
        """Full XML element representing the styles information."""

        self.para_style_dict: dict[str, DocxStyle] = {}
        """Dictionary mapping style IDs to their definitions."""

        if styles_xml is not None:
            # Find all styles and create dictionary
            for style in styles_xml.findall(
                ".//w:style",
                namespaces=docx_namespaces,
            ):
                style_id = style.attrib.get(
                    f"{{{docx_namespaces['w']}}}styleId",
                )
                if style_id is not None:
                    style_instance = DocxStyle(style)
                    if style_instance.style_type == "paragraph":
                        self.para_style_dict[style_id] = style_instance

    def __repr__(self) -> str:
        """Return a string representation of the StyleParts instance."""
        return f"StyleParts(style_count={len(self.para_style_dict)})"


class DocxStyle:
    """Represent a style in a DOCX document."""

    def __init__(self, style_xml: etree._Element) -> None:
        """
        Initialize the DocxStyle with the XML element representing the style.

        Args:
            style_xml (etree.Element): The XML element representing the style.

        """
        self.style_xml = style_xml
        self.level: int = 0
        self.num_id: int | None = None
        self.style_type = style_xml.attrib.get(
            f"{{{docx_namespaces['w']}}}type",
            "paragraph",
        )
        self.style_id = style_xml.attrib.get(
            f"{{{docx_namespaces['w']}}}styleId",
            "MISSING",
        )

        # Find child <w:numPr> found under <w:pPr>/<w:numPr>
        for num_pr in style_xml.findall(
            ".//w:numPr",
            namespaces=docx_namespaces,
        ):
            self.level = get_xml_child_int(num_pr, "w:ilvl") or 0
            self.num_id = get_xml_child_int(num_pr, "w:numId")

    def __repr__(self) -> str:
        """Return a string representation of the DocxStyle instance."""
        return f"DocxStyle(style_id={self.style_id}/Level={self.level}/NumId={self.num_id})"
