"""
Class to track the numbering of items in a document.

This class is used to keep track of the numbering of items in a document, such as lists or headings.
Tracking the numbering is important for maintaining the correct order and reference to items in the document.
The class provides methods to add items and retrieve their assigned numbers.

The number formats include:
- 1, 2, 3, ... (for lists & headings)
- 1.1, 1.2, 1.3, ... (for subheadings)
- a, b, c, ... (for sublists)
- i, ii, iii, ... (for headings)
- A, B, C, ... (for subheadings)
"""

import logging
from collections import defaultdict
from typing import Dict, List

logger = logging.getLogger(__name__)


class NumberPartFormat:
    """
    Enum-like class to define the formats of numbered items.

    Attributes:
        BULLET (str): Represents a bullet format.
        HYPHEN (str): Represents a hyphen format.
        NUMERIC (str): Represents a numeric format.
        LOWER_ALPHABETIC (str): Represents a lowercase alphabetic format.
        UPPER_ALPHABETIC (str): Represents an uppercase alphabetic format.
        LOWER_ROMAN (str): Represents a lowercase roman numeral format.
        UPPER_ROMAN (str): Represents an uppercase roman numeral format.
        DECIMAL (str): Represents a decimal format.
        DECIMAL_LEADING_ZERO (str): Represents a decimal format with leading zero.

    """

    BULLET = "bullet"
    HYPHEN = "hyphen"
    ALPHABETIC = "lowerLetter"
    UPPER_ALPHABETIC = "upperLetter"
    ROMAN = "lowerRoman"
    UPPER_ROMAN = "upperRoman"
    DECIMAL = "decimal"
    DECIMAL_LEADING_ZERO = "decimal_leading_zero"
    DECIMAL_ZERO = "decimalZero"


class TrackerType:
    """
    Enum-like class to define the types of trackers available.

    Attributes:
        LIST (str): Represents a list tracker.
        HEADING (str): Represents a heading tracker.
        SUBLIST (str): Represents a sublist tracker.
        SUBHEADING (str): Represents a subheading tracker.

    """

    FULLY_DECORATED = "fully_decorated"
    TEMPLATE_LEVEL = "template_level"
    LEVEL_ONLY = "level"
    MD_LIST = "md_list"
    HTML_LIST = "html_list"
    MD_HEADING = "md_heading"
    MD_HEADING_NUMBERED = "md_numbered_heading"


class ListNumberTracker:
    """
    Class to track numbering for headers and lists on a per-level basis.

    This class generates hierarchical numbering (e.g., 1, 1.1, 1.2, etc.)
    based on the level provided. It supports customization of numbering formats
    and tracker types.
    """

    def __init__(
        self,
        tracker_type: str,
        formats: List[str],
        template_formats: List[str] | None = None,
        default_starting_numbers: List[int] | None = None,
    ) -> None:
        """
        Initialize the tracker with a specific tracker type and numbering formats.

        Args:
            tracker_type (str): The type of tracker (e.g., "fully_decorated", "level").
            formats (List[str]): A list of formats for each level (e.g., ["numeric", "lower_alphabetic"]).
            template_formats (List[str]): A list of template formats for each level.
            default_starting_numbers (List[int]): A list of starting numbers for each level.

        """
        self.tracker_type = tracker_type
        self.formats = formats
        self.template_formats = template_formats or []
        """Mapping of level numbers to their template formats."""

        self._level_numbers: Dict[int, int] = defaultdict(int)
        """Mapping of level numbers to their current values."""

        self._level_start_numbers: Dict[int, int] = defaultdict(int)
        """Mapping of level numbers to their starting values."""

        if default_starting_numbers:
            for idx, start_val in enumerate(default_starting_numbers):
                self._level_start_numbers[idx] = start_val
        logger.debug(self.__repr__())

    # display class contents
    def __repr__(self) -> str:
        """Return a string representation of the ListNumberTracker instance."""
        return f"ListNumberTracker(tracker_type={self.tracker_type}, formats={self.formats}, template_formats={self.template_formats})"

    def get_number(self, level: int, list_text: str | None = None) -> str:
        """
        Get the next number for the given level.

        Args:
            level (int): The level of the header or list.
            list_text (str | None): Optional text for the list item.

        Returns:
            str: The hierarchical number for the given level.

        Exceptions:
            ValueError: If the tracker type is not supported.

        """
        # Ensure parent levels (0 .. level-1) exist with their starting values
        # so a call to a deeper level before parents still renders like "1.1"
        for parent in range(level):
            if self._level_numbers[parent] == 0:
                # set parent to its starting value (do not subtract 1)
                self._level_numbers[parent] = self._level_start_numbers.get(parent, 1)

        # Check if this is a new level, if so start at the requested number
        # Add value -1 to ensure we start at the correct number
        if self._level_numbers[level] == 0:
            self._level_numbers[level] = self._level_start_numbers.get(level, 1) - 1

        # Increment the current level's number
        self._level_numbers[level] += 1

        # Reset numbers for deeper levels
        for deeper_level in range(level + 1, max(self._level_numbers.keys()) + 1):
            self._level_numbers[deeper_level] = 0

        # Generate the hierarchical number
        number_parts = [
            self._format_number(self._level_numbers[lvl], lvl)
            for lvl in sorted(self._level_numbers.keys())
            if self._level_numbers[lvl] > 0 and lvl <= level
        ]

        indent_str = "  " * level if self.tracker_type == TrackerType.MD_LIST else ""
        logger.debug(
            f"get_number(L[{level}], Indent:{len(indent_str)}, Type: {self.tracker_type})",
        )

        if self.tracker_type == TrackerType.TEMPLATE_LEVEL:
            list_part = self._apply_template_formats(
                number_parts,
                self.template_formats[level],
            )
        elif self.tracker_type == TrackerType.FULLY_DECORATED:
            list_part = ".".join(number_parts)
        elif self.tracker_type == TrackerType.LEVEL_ONLY:
            list_part = number_parts[-1]  # Only return the current level's number
        elif self.tracker_type == TrackerType.MD_LIST:
            list_part = number_parts[-1]
            if number_parts[-1] not in ["*", "-"]:
                list_part += "."
        else:
            raise ValueError("Unsupported tracker type: %s", self.tracker_type)

        return_val = f"{indent_str}{list_part}{' ' if list_text else ''}{list_text if list_text else ''}"
        logger.debug(f"Generated tracker text: [{return_val}]")
        return return_val

    def _format_number(self, number: int, level: int) -> str:  # noqa: PLR0911
        """
        Format the number based on the specified format type.

        Args:
            number (int): The number to format.
            level (int): The level of the formatting.

        Returns:
            str: The formatted number.

        Exceptions:
            ValueError: If the tracker type is not supported.

        """
        if level > len(self.formats) - 1:
            format_type = NumberPartFormat.BULLET
        else:
            format_type = self.formats[level]
        logger.debug(
            f"Formatting number: {number} with level[{level}] type: {format_type}",
        )

        if format_type in [
            NumberPartFormat.DECIMAL,
            NumberPartFormat.DECIMAL_ZERO,
            NumberPartFormat.DECIMAL_LEADING_ZERO,
        ]:
            return str(number)
        if format_type == NumberPartFormat.ALPHABETIC:
            return chr(96 + number)  # Convert to 'a', 'b', 'c', etc.
        if format_type == NumberPartFormat.UPPER_ALPHABETIC:
            return chr(64 + number)  # Convert to 'A', 'B', 'C', etc.
        if format_type == NumberPartFormat.ROMAN:
            return self._to_roman(number).lower()
        if format_type == NumberPartFormat.UPPER_ROMAN:
            return self._to_roman(number).upper()
        if format_type == NumberPartFormat.BULLET:
            return "*"
        if format_type == NumberPartFormat.HYPHEN:
            return "-"
        raise ValueError("Unsupported format type: %s", format_type)

    def _to_roman(self, number: int) -> str:
        """
        Convert an integer to a Roman numeral.

        Args:
            number (int): The number to convert.

        Returns:
            str: The Roman numeral representation.

        """
        roman_numerals = {
            1: "I",
            4: "IV",
            5: "V",
            9: "IX",
            10: "X",
            40: "XL",
            50: "L",
            90: "XC",
            100: "C",
            400: "CD",
            500: "D",
            900: "CM",
            1000: "M",
        }
        result = ""
        for value, numeral in sorted(roman_numerals.items(), key=lambda x: -x[0]):
            while number >= value:
                result += numeral
                number -= value
        return result

    def reset(self) -> None:
        """
        Reset the tracker to start numbering from the beginning.

        This is useful when starting a new section or document.
        """
        self._level_numbers.clear()

    def _apply_template_formats(
        self,
        number_parts: List[str],
        template_format: str,
    ) -> str:
        """
        Apply the template format to the number parts.

        Args:
            number_parts (List[str]): The list of formatted number parts.
            template_format (str): The template format string.

        Returns:
            str: The formatted string according to the template.

        """
        # Simple implementation: replace placeholders {n} with number_parts[n]
        result = template_format
        for idx, part in enumerate(number_parts):
            result = result.replace(f"%{idx + 1}", part)
        return result
