"""
Unit tests for the exclude-colors CLI functionality.

This module contains test cases for validating hex color codes and testing
the --exclude-colors command-line argument functionality.
"""

import unittest
from unittest.mock import patch

from kyd_docx2md.tools.cli_kyd_docx2md import parse_args, validate_hex_colors


class TestValidateHexColors(unittest.TestCase):
    """Test cases for the validate_hex_colors function."""

    def test_valid_single_color(self) -> None:
        """Test validation of a single valid hex color."""
        result = validate_hex_colors("#FFFFFF")
        self.assertEqual(result, ["FFFFFF"])

    def test_valid_multiple_colors(self) -> None:
        """Test validation of multiple valid hex colors."""
        result = validate_hex_colors("#FFFFFF,#000000,#FF5733")
        self.assertEqual(result, ["FFFFFF", "000000", "FF5733"])

    def test_valid_lowercase_colors(self) -> None:
        """Test that lowercase hex colors are converted to uppercase."""
        result = validate_hex_colors("#ffffff,#aabbcc")
        self.assertEqual(result, ["FFFFFF", "AABBCC"])

    def test_valid_mixed_case_colors(self) -> None:
        """Test that mixed case hex colors are converted to uppercase."""
        result = validate_hex_colors("#FfFfFf,#AaBbCc")
        self.assertEqual(result, ["FFFFFF", "AABBCC"])

    def test_empty_string(self) -> None:
        """Test validation of an empty string."""
        result = validate_hex_colors("")
        self.assertEqual(result, [])

    def test_none_string(self) -> None:
        """Test validation of None."""
        result = validate_hex_colors(None)
        self.assertEqual(result, [])

    def test_invalid_format_no_hash(self) -> None:
        """Test that colors without # are rejected."""
        with self.assertRaises(ValueError) as context:
            validate_hex_colors("FFFFFF")
        self.assertIn("Invalid hex color format", str(context.exception))

    def test_invalid_format_short(self) -> None:
        """Test that colors with less than 6 hex digits are rejected."""
        with self.assertRaises(ValueError) as context:
            validate_hex_colors("#FFF")
        self.assertIn("Invalid hex color format", str(context.exception))

    def test_invalid_format_long(self) -> None:
        """Test that colors with more than 6 hex digits are rejected."""
        with self.assertRaises(ValueError) as context:
            validate_hex_colors("#FFFFFFF")
        self.assertIn("Invalid hex color format", str(context.exception))

    def test_invalid_format_non_hex(self) -> None:
        """Test that colors with non-hex characters are rejected."""
        with self.assertRaises(ValueError) as context:
            validate_hex_colors("#GGGGGG")
        self.assertIn("Invalid hex color format", str(context.exception))

    def test_mixed_valid_invalid(self) -> None:
        """Test that mixed valid/invalid colors are rejected with error message."""
        with self.assertRaises(ValueError) as context:
            validate_hex_colors("#FFFFFF,#GGG,#000000")
        self.assertIn("Invalid hex color format", str(context.exception))

    def test_extra_commas(self) -> None:
        """Test that extra commas are handled gracefully."""
        result = validate_hex_colors("#FFFFFF,,#000000,")
        self.assertEqual(result, ["FFFFFF", "000000"])

    def test_whitespace_handling(self) -> None:
        """Test that whitespace around colors is handled."""
        result = validate_hex_colors(" #FFFFFF , #000000 ")
        self.assertEqual(result, ["FFFFFF", "000000"])


class TestParseArgsExcludeColors(unittest.TestCase):
    """Test cases for the --exclude-colors argument parsing."""

    @patch(
        "sys.argv",
        ["cli_kyd_docx2md", "test.docx", "--exclude-colors", "#FFFFFF,#000000"],
    )
    def test_exclude_colors_argument(self) -> None:
        """Test that --exclude-colors argument is parsed correctly."""
        args = parse_args()
        self.assertEqual(args.exclude_colors, ["FFFFFF", "000000"])

    @patch("sys.argv", ["cli_kyd_docx2md", "test.docx", "-ec", "#FF5733"])
    def test_exclude_colors_short_argument(self) -> None:
        """Test that -ec short argument is parsed correctly."""
        args = parse_args()
        self.assertEqual(args.exclude_colors, ["FF5733"])

    @patch("sys.argv", ["cli_kyd_docx2md", "test.docx"])
    def test_no_exclude_colors_argument(self) -> None:
        """Test that missing --exclude-colors argument results in None."""
        args = parse_args()
        self.assertIsNone(args.exclude_colors)

    @patch("sys.argv", ["cli_kyd_docx2md", "test.docx", "--exclude-colors", "#INVALID"])
    def test_invalid_exclude_colors_raises_error(self) -> None:
        """Test that invalid hex colors raise ValueError."""
        with self.assertRaises(ValueError) as context:
            parse_args()
        self.assertIn("Invalid hex color format", str(context.exception))

    @patch(
        "sys.argv",
        ["cli_kyd_docx2md", "test.docx", "--exclude-colors", "#FFFFFF,#GGG"],
    )
    def test_mixed_invalid_exclude_colors_raises_error(self) -> None:
        """Test that mixed valid/invalid colors raise ValueError."""
        with self.assertRaises(ValueError) as context:
            parse_args()
        self.assertIn("Invalid hex color format", str(context.exception))


if __name__ == "__main__":
    unittest.main()
