"""Unit tests for the kyd_docx2md CLI tool."""

import argparse
import unittest
from pathlib import Path
from unittest.mock import MagicMock, mock_open, patch

from kyd_docx2md.tools.cli_kyd_docx2md import (
    determine_output_files,
    expand_inputs,
    filter_existing_files,
    load_custom_style,
    main,
    parse_args,
    process_files,
)


class TestCliKydDocx2Md(unittest.TestCase):
    """Test cases for the kyd_docx2md CLI tool."""

    @patch(
        "sys.argv",
        [
            "cli_kyd_docx2md",
            "test.docx",
            "--image-folder",
            "test_images",
            "--output",
            "output.md",
            "--ascii_only",
            "--no-images",
            "--remove-wrapping-tables",
            "1x1",
            "2x2",
            "--log-level",
            "DEBUG",
            "--log-output",
            "test.log",
            "--custom-style",
            "style.json",
            "--exclude-colors",
            "#FFFFFF",
        ],
    )
    def test_parse_args_all_args(self) -> None:
        """Test parsing of all command-line arguments."""
        args = parse_args()
        self.assertEqual(args.inputs, ["test.docx"])
        self.assertEqual(args.image_folder, "test_images")
        self.assertEqual(args.output, "output.md")
        self.assertTrue(args.ascii_only)
        self.assertFalse(args.export_images)
        self.assertEqual(args.remove_wrapping_tables, ["1x1", "2x2"])
        self.assertEqual(args.log_level, "DEBUG")
        self.assertEqual(args.log_output, "test.log")
        self.assertEqual(args.custom_style, "style.json")
        self.assertEqual(args.exclude_colors, ["FFFFFF"])

    @patch("sys.argv", ["cli_kyd_docx2md", "test.docx"])
    def test_parse_args_defaults(self) -> None:
        """Test default values of command-line arguments."""
        args = parse_args()
        self.assertEqual(args.inputs, ["test.docx"])
        self.assertEqual(args.image_folder, "./images")
        self.assertIsNone(args.output)
        self.assertFalse(args.ascii_only)
        self.assertTrue(args.export_images)
        self.assertIsNone(args.remove_wrapping_tables)
        self.assertEqual(args.log_level, "WARNING")
        self.assertIsNone(args.log_output)
        self.assertIsNone(args.custom_style)
        self.assertIsNone(args.exclude_colors)

    @patch("sys.argv", ["cli_kyd_docx2md", "test.docx", "--remove-wrapping-tables"])
    def test_parse_args_remove_wrapping_tables_default(self) -> None:
        """Test default value for --remove-wrapping-tables when flag is present."""
        args = parse_args()
        self.assertEqual(args.remove_wrapping_tables, ["1x1"])

    @patch(
        "sys.argv",
        [
            "cli_kyd_docx2md",
            "test.docx",
            "--remove-wrapping-tables",
            "1x1",
            "invalid",
            "2x2",
            "0x1",
        ],
    )
    def test_parse_args_remove_wrapping_tables_invalid_sizes(self) -> None:
        """Test that invalid sizes for --remove-wrapping-tables are ignored."""
        with self.assertLogs(
            "kyd_docx2md.tools.cli_kyd_docx2md",
            level="WARNING",
        ) as cm:
            args = parse_args()
            self.assertEqual(args.remove_wrapping_tables, ["1x1", "2x2"])
            self.assertIn("Ignoring invalid table size: invalid", cm.output[0])
            self.assertIn("Ignoring invalid table size: 0x1", cm.output[1])

    @patch("pathlib.Path.expanduser")
    @patch("pathlib.Path.glob")
    def test_expand_inputs_with_glob(
        self,
        mock_glob: MagicMock,
        mock_expanduser: MagicMock,
    ) -> None:
        """Test input expansion with glob patterns."""
        mock_expanduser.return_value = Path("/home/user/docs/*.docx")
        mock_glob.return_value = [
            Path("/home/user/docs/doc1.docx"),
            Path("/home/user/docs/doc2.docx"),
        ]

        inputs = ["~/docs/*.docx"]
        result = expand_inputs(inputs)
        self.assertEqual(len(result), 2)
        self.assertIn(str(Path("/home/user/docs/doc1.docx")), result)
        self.assertIn(str(Path("/home/user/docs/doc2.docx")), result)

    @patch("pathlib.Path.exists", return_value=True)
    def test_expand_inputs_existing_file(self, mock_exists: MagicMock) -> None:
        """Test input expansion with an existing file."""
        inputs = ["test.docx"]
        result = expand_inputs(inputs)
        self.assertEqual(result, ["test.docx"])
        mock_exists.assert_called()

    @patch("pathlib.Path.exists", return_value=False)
    def test_expand_inputs_non_existing_file(self, mock_exists: MagicMock) -> None:
        """Test input expansion with a non-existing file."""
        inputs = ["non-existent.docx"]
        result = expand_inputs(inputs)
        self.assertEqual(result, ["non-existent.docx"])
        mock_exists.assert_called()

    @patch("pathlib.Path.exists", side_effect=[True, False, True])
    def test_filter_existing_files(self, mock_exists: MagicMock) -> None:
        """Test filtering of existing files."""
        files = ["a.docx", "b.docx", "c.docx"]
        with self.assertLogs(
            "kyd_docx2md.tools.cli_kyd_docx2md",
            level="WARNING",
        ) as cm:
            result = filter_existing_files(files)
            self.assertEqual(result, [str(Path("a.docx")), str(Path("c.docx"))])
            self.assertIn("Input file not found/skipped: b.docx", cm.output[0])
        self.assertEqual(mock_exists.call_count, 3)

    def test_determine_output_files_single_file_no_output(self) -> None:
        """Test output file determination for a single file with no output specified."""
        with patch("pathlib.Path.mkdir"):
            result = determine_output_files(["test.docx"], "")
            self.assertEqual(result, [str(Path("test.md"))])

    def test_determine_output_files_multiple_files_no_output(self) -> None:
        """Test output file determination for multiple files with no output specified."""
        with patch("pathlib.Path.mkdir"):
            result = determine_output_files(["a.docx", "b.docx"], "")
            self.assertEqual(result, [str(Path("a.md")), str(Path("b.md"))])

    def test_determine_output_files_single_file_with_output_file(self) -> None:
        """Test output file determination for a single file with an output file specified."""
        with patch("pathlib.Path.mkdir"):
            result = determine_output_files(["test.docx"], "output.md")
            self.assertEqual(result, ["output.md"])

    def test_determine_output_files_multiple_files_with_output_dir(self) -> None:
        """Test output file determination for multiple files with an output directory specified."""
        with patch("pathlib.Path.mkdir"):
            result = determine_output_files(["a.docx", "b.docx"], "out_dir")
            self.assertEqual(
                result,
                [str(Path("out_dir/a.md")), str(Path("out_dir/b.md"))],
            )

    @patch("builtins.open", new_callable=mock_open, read_data='{"style": "custom"}')
    @patch("pathlib.Path.exists", return_value=True)
    def test_load_custom_style_valid(
        self,
        mock_exists: MagicMock,
        mock_file: MagicMock,
    ) -> None:
        """Test loading a valid custom style file."""
        result = load_custom_style("style.json")
        self.assertEqual(result, {"style": "custom"})
        mock_exists.assert_called()
        mock_file.assert_called()

    @patch("pathlib.Path.exists", return_value=False)
    def test_load_custom_style_not_found(self, mock_exists: MagicMock) -> None:
        """Test loading a non-existent custom style file."""
        with self.assertLogs("kyd_docx2md.tools.cli_kyd_docx2md", level="ERROR") as cm:
            result = load_custom_style("non-existent.json")
            self.assertIsNone(result)
            self.assertIn("Custom style file does not exist", cm.output[0])
        mock_exists.assert_called()

    def test_load_custom_style_not_json(self) -> None:
        """Test loading a file that is not a JSON file."""
        with self.assertLogs("kyd_docx2md.tools.cli_kyd_docx2md", level="ERROR") as cm:
            result = load_custom_style("style.txt")
            self.assertIsNone(result)
            self.assertIn("Custom style file must be a JSON file", cm.output[0])
        # Note: exists() is not called when file doesn't end with .json

    @patch("builtins.open", new_callable=mock_open, read_data="invalid json")
    @patch("pathlib.Path.exists", return_value=True)
    def test_load_custom_style_invalid_json(
        self,
        mock_exists: MagicMock,
        mock_file: MagicMock,
    ) -> None:
        """Test loading an invalid JSON custom style file."""
        with self.assertLogs("kyd_docx2md.tools.cli_kyd_docx2md", level="ERROR") as cm:
            result = load_custom_style("style.json")
            self.assertIsNone(result)
            self.assertIn("Invalid JSON format in custom style file", cm.output[0])
        mock_exists.assert_called()
        mock_file.assert_called()

    @patch("kyd_docx2md.tools.cli_kyd_docx2md.Docx2Md")
    def test_process_files(self, mock_docx2md: MagicMock) -> None:
        """Test the file processing function."""
        args = argparse.Namespace(
            ascii_only=False,
            image_folder="images",
            export_images=True,
            exclude_colors=None,
            remove_wrapping_tables=None,
            custom_style=None,
        )
        mock_instance = mock_docx2md.return_value
        process_files(["test.docx"], ["test.md"], args, None)
        mock_docx2md.assert_called_once()
        mock_instance.convert_docx_2_md.assert_called_with("test.md")

    @patch("kyd_docx2md.tools.cli_kyd_docx2md.Docx2Md")
    def test_wrapping_tables(self, mock_docx2md: MagicMock) -> None:
        """Test the wrapping tables option."""
        args = argparse.Namespace(
            ascii_only=False,
            image_folder="images",
            export_images=True,
            exclude_colors=["FF5733"],
            remove_wrapping_tables=["1x1", "-2x2", "2b2"],
            custom_style=None,
        )
        mock_instance = mock_docx2md.return_value
        process_files(["test.docx"], ["test.md"], args, None)
        mock_docx2md.assert_called_once()
        mock_instance.convert_docx_2_md.assert_called_with("test.md")

    @patch("kyd_docx2md.tools.cli_kyd_docx2md.parse_args")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.expand_inputs")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.filter_existing_files")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.determine_output_files")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.load_custom_style")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.process_files")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.setup_logger")
    def test_main(  # noqa: PLR0913
        self,
        mock_setup_logger: MagicMock,
        mock_process_files: MagicMock,
        mock_load_custom_style: MagicMock,
        mock_determine_output_files: MagicMock,
        mock_filter_existing_files: MagicMock,
        mock_expand_inputs: MagicMock,
        mock_parse_args: MagicMock,
    ) -> None:
        """Test the main function."""
        mock_args = argparse.Namespace(
            inputs=["test.docx"],
            output=None,
            custom_style=None,
            log_level="INFO",
            log_output=None,
        )
        mock_parse_args.return_value = mock_args
        mock_expand_inputs.return_value = ["test.docx"]
        mock_filter_existing_files.return_value = ["test.docx"]
        mock_determine_output_files.return_value = ["test.md"]
        mock_load_custom_style.return_value = None

        main()

        mock_parse_args.assert_called_once()
        mock_expand_inputs.assert_called_with(["test.docx"])
        mock_setup_logger.assert_called_with(log_level="INFO", log_fileName=None)
        mock_filter_existing_files.assert_called_with(["test.docx"])
        mock_determine_output_files.assert_called_with(["test.docx"], None)
        mock_load_custom_style.assert_not_called()
        mock_process_files.assert_called_once()

    @patch("kyd_docx2md.tools.cli_kyd_docx2md.parse_args")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.expand_inputs")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.filter_existing_files")
    @patch("kyd_docx2md.tools.cli_kyd_docx2md.setup_logger")
    def test_main_no_input_files(
        self,
        mock_setup_logger: MagicMock,
        mock_filter_existing_files: MagicMock,
        mock_expand_inputs: MagicMock,
        mock_parse_args: MagicMock,
    ) -> None:
        """Test main function with no valid input files."""
        mock_args = argparse.Namespace(
            inputs=[],
            log_level="INFO",
            log_output=None,
        )
        mock_parse_args.return_value = mock_args
        mock_expand_inputs.return_value = []
        mock_filter_existing_files.return_value = []
        with self.assertLogs("kyd_docx2md.tools.cli_kyd_docx2md", level="ERROR") as cm:
            main()
            self.assertIn("No valid input DOCX files were provided.", cm.output[0])

        mock_parse_args.assert_called_once()
        mock_expand_inputs.assert_called_with([])
        mock_setup_logger.assert_called_with(log_level="INFO", log_fileName=None)
        mock_filter_existing_files.assert_called_with([])


if __name__ == "__main__":
    unittest.main()
