"""
Unit tests for the kyd_docx2md module.

This module contains test cases for the Docx2Md class, which converts .docx files
to markdown format. It includes tests for various features such as tables, images,
styles, hyperlinks, and text formatting.
"""

import unittest
from pathlib import Path

import pytest
from dotenv import load_dotenv
import re

from kyd_docx2md.numbered_item_tracker import (
    ListNumberTracker,
    NumberPartFormat,
    TrackerType,
)

load_dotenv()  # take environment variables from .env.


test_data_dir = Path(__file__).parent / "test-data"

# Use temporary directory for test data


class TestNumbering(unittest.TestCase):
    """Unit tests for numbering and list tracking in the kyd_docx2md module."""

    def test_number_part_constants(self) -> None:
        """Test that NumberPartFormat constants have the expected string values."""
        assert NumberPartFormat.BULLET == "bullet"
        assert NumberPartFormat.HYPHEN == "hyphen"
        assert NumberPartFormat.ALPHABETIC == "lowerLetter"
        assert NumberPartFormat.UPPER_ALPHABETIC == "upperLetter"
        assert NumberPartFormat.ROMAN == "lowerRoman"
        assert NumberPartFormat.UPPER_ROMAN == "upperRoman"
        assert NumberPartFormat.DECIMAL == "decimal"
        assert NumberPartFormat.DECIMAL_LEADING_ZERO == "decimal_leading_zero"

    def test_list_tracker_type_constants(self) -> None:
        """Test that TrackerType constants have the expected string values."""
        assert TrackerType.FULLY_DECORATED == "fully_decorated"
        assert TrackerType.LEVEL_ONLY == "level"
        assert TrackerType.MD_LIST == "md_list"
        assert TrackerType.HTML_LIST == "html_list"
        assert TrackerType.MD_HEADING == "md_heading"

    def test_list_tracker_ex_1(self) -> None:
        """Test ListNumberTracker with numeric and lower alphabetic formats."""
        tracker = ListNumberTracker(
            TrackerType.LEVEL_ONLY,
            [NumberPartFormat.DECIMAL, NumberPartFormat.ALPHABETIC],
        )
        assert tracker.get_number(0) == "1"
        assert tracker.get_number(1) == "a"
        assert tracker.get_number(1) == "b"
        assert tracker.get_number(0) == "2"
        assert tracker.get_number(1) == "a"

    def test_list_tracker_ex_2(self) -> None:
        """Test ListNumberTracker with repeated calls and expected output."""
        tracker = ListNumberTracker(
            TrackerType.FULLY_DECORATED,
            [
                NumberPartFormat.DECIMAL,
                NumberPartFormat.ROMAN,
                NumberPartFormat.UPPER_ROMAN,
                NumberPartFormat.ALPHABETIC,
                NumberPartFormat.UPPER_ALPHABETIC,
                NumberPartFormat.DECIMAL,
            ],
        )

        tracker.reset()

        assert tracker.get_number(0) == "1"
        assert tracker.get_number(1) == "1.i"
        assert tracker.get_number(2) == "1.i.I"
        assert tracker.get_number(3) == "1.i.I.a"
        assert tracker.get_number(4) == "1.i.I.a.A"
        assert tracker.get_number(5) == "1.i.I.a.A.1"

        assert tracker.get_number(0) == "2"
        assert tracker.get_number(1) == "2.i"
        assert tracker.get_number(2) == "2.i.I"
        assert tracker.get_number(3) == "2.i.I.a"
        assert tracker.get_number(4) == "2.i.I.a.A"
        assert tracker.get_number(5) == "2.i.I.a.A.1"

    def test_list_tracker_ex_3(self) -> None:
        """Test ListNumberTracker with markdown list formats and indentation."""
        tracker = ListNumberTracker(
            TrackerType.MD_LIST,
            [
                NumberPartFormat.DECIMAL,
                NumberPartFormat.ROMAN,
                NumberPartFormat.UPPER_ROMAN,
                NumberPartFormat.ALPHABETIC,
                NumberPartFormat.UPPER_ALPHABETIC,
                NumberPartFormat.DECIMAL,
                NumberPartFormat.BULLET,
                NumberPartFormat.HYPHEN,
            ],
        )

        tracker.reset()

        assert tracker.get_number(0) == "1."
        assert tracker.get_number(1) == "  i."
        assert tracker.get_number(2) == "    I."
        assert tracker.get_number(3) == "      a."
        assert tracker.get_number(4) == "        A."
        assert tracker.get_number(5) == "          1."
        assert tracker.get_number(6) == "            *"
        assert tracker.get_number(7) == "              -"

    def test_header_and_item_number_tracker_reset(self) -> None:
        """Test ListNumberTracker reset functionality."""
        tracker = ListNumberTracker(
            "fully_decorated",
            [NumberPartFormat.DECIMAL, NumberPartFormat.ALPHABETIC],
        )
        assert tracker.get_number(0) == "1"
        assert tracker.get_number(1) == "1.a"
        tracker.reset()
        assert tracker.get_number(0) == "1"
        assert tracker.get_number(1) == "1.a"

    def test_markdown_list_tracker(self) -> None:
        """Test markdown list generation functionality."""
        tracker = ListNumberTracker(
            TrackerType.MD_LIST,
            ["bullet", "bullet"],
        )
        assert tracker.get_number(0) == "*"
        assert tracker.get_number(1) == "  *"
        assert tracker.get_number(1) == "  *"
        assert tracker.get_number(0) == "*"

    def test_bad_tracker_type(self) -> None:
        """Test ListNumberTracker reset functionality."""
        # Test bad tracker type throws the expected exception
        tracker = ListNumberTracker(
            "bad_tracker_type",
            [NumberPartFormat.DECIMAL, NumberPartFormat.ALPHABETIC],
        )
        with pytest.raises(
            ValueError,
            match=re.escape("('Unsupported tracker type: %s', 'bad_tracker_type')"),
        ):
            tracker.get_number(1)

    def test_bad_formats(self) -> None:
        """Test ListNumberTracker reset functionality."""
        # Test bad tracker type throws the expected exception
        tracker = ListNumberTracker(
            "fully_decorated",
            [NumberPartFormat.DECIMAL, "bad_alpha"],
        )
        tracker.get_number(0)
        with pytest.raises(
            ValueError,
            match=re.escape("('Unsupported format type: %s', 'bad_alpha')"),
        ):
            tracker.get_number(1)

    def test_docx_templates(self) -> None:
        """Test ListNumberTracker with DOCX template formats."""
        tracker = ListNumberTracker(
            TrackerType.TEMPLATE_LEVEL,
            [
                NumberPartFormat.DECIMAL,
                NumberPartFormat.ALPHABETIC,
                NumberPartFormat.UPPER_ALPHABETIC,
            ],
            template_formats=[
                "%1.",
                "%1.%2.",
                "%1.%2.%3.",
            ],
        )

        assert tracker.get_number(0) == "1."
        assert tracker.get_number(1) == "1.a."
        assert tracker.get_number(2) == "1.a.A."
        assert tracker.get_number(0) == "2."
        assert tracker.get_number(1) == "2.a."
        assert tracker.get_number(1) == "2.b."
        assert tracker.get_number(2) == "2.b.A."

    def test_docx_start_numbering(self) -> None:
        """Test numbering starts at the specified values."""
        tracker = ListNumberTracker(
            TrackerType.TEMPLATE_LEVEL,
            [
                NumberPartFormat.DECIMAL,
                NumberPartFormat.ALPHABETIC,
                NumberPartFormat.UPPER_ALPHABETIC,
            ],
            [
                "%1.",
                "%1.%2.",
                "%1.%2.%3.",
            ],
            [4, 5, 6],
        )

        assert tracker.get_number(0) == "4."
        assert tracker.get_number(1) == "4.e."
        assert tracker.get_number(2) == "4.e.F."
        assert tracker.get_number(0) == "5."
        assert tracker.get_number(1) == "5.e."
        assert tracker.get_number(1) == "5.f."
        assert tracker.get_number(2) == "5.f.F."

    def test_docx_start_missed_levels(self) -> None:
        """Tests if a sequence is not started at level 0 works as expected."""
        tracker = ListNumberTracker(
            TrackerType.TEMPLATE_LEVEL,
            [
                NumberPartFormat.DECIMAL,
                NumberPartFormat.ALPHABETIC,
                NumberPartFormat.UPPER_ALPHABETIC,
            ],
            [
                "%1.",
                "%1.%2.",
                "%1.%2.%3.",
            ],
            [4, 1, 1],
        )

        assert tracker.get_number(1) == "4.a."
